using System;
using System.Data;
using MethodBase = System.Reflection.MethodBase;
using System.Collections;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary></summary>

		#endregion

	public class ReagentType : BaseBusinessObject
	{
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1267"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid ReagentType object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1093"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Empty constructor
		/// </summary>
		public ReagentType()
		{
			_expectedMinReactions = BOL.ReagentType.GetCanChangeExpectedMinReactions();
			for(int nCtr = 0; nCtr < _expectedMinReactions.Count; nCtr ++)
			{
				_expectedMinReactions[nCtr] = _expectedMinReactions[nCtr].ToString().Replace("+","").Trim();
			}
		}
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1268"> 
		///		<ExpectedInput>Valid ReagentTypeId, ReagentTypeName and minimumLevel (short)</ExpectedInput>
		///		<ExpectedOutput>Valid ReagentType object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1166"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ReagentType constructor
		/// </summary>
		public ReagentType(int reagentTypeID, string reagentTypeName, short minimumLevel)
		{
			_expectedMinReactions = BOL.ReagentType.GetCanChangeExpectedMinReactions();
			for(int nCtr = 0; nCtr < _expectedMinReactions.Count; nCtr ++)
			{
				_expectedMinReactions[nCtr] = _expectedMinReactions[nCtr].ToString().Replace("+","").Trim();
			}
			this._reagentTypeID = reagentTypeID;
			this._reagentTypeName = reagentTypeName;
			this._minimumStockLevel = minimumLevel;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2420"> 
		///		<ExpectedInput>DataRow containing ReagentType data</ExpectedInput>
		///		<ExpectedOutput>ReagentType object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2874"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///		
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates ReagentType object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public ReagentType(DataRow dr)
		{
			_expectedMinReactions = BOL.ReagentType.GetCanChangeExpectedMinReactions();
			for(int nCtr = 0; nCtr < _expectedMinReactions.Count; nCtr ++)
			{
				_expectedMinReactions[nCtr] = _expectedMinReactions[nCtr].ToString().Replace("+","").Trim();
			}
			this.LoadFromDataRow(dr);
		}

		#region Variables
		
		///	<summary>
		///	ID of reagent.
		///	</summary>
		private int _reagentTypeID;
		///	<summary>
		///	Name of reagent.
		///	</summary>
		private string _reagentTypeName;
		///	<summary>
		///	Minimum stock level.
		///	</summary>
		private short _minimumStockLevel;
		///	<summary>
		///	Sort Column.
		///	</summary>
		private short _sortColumn;
		///	<summary>
		///	Indicator that says this reagent type must be QC daily.
		///	</summary>
		private bool _mustUseIndicator;
		/// <summary>
		/// Indicator that says this reagent type must be used today
		/// </summary>
		private bool _dailyUseIndicator;
		/// <summary>
		/// Indicator that says the expected min reaction can be changed
		/// </summary>
		private bool _canChangeIndicator;
		/// <summary>
		/// Char 1 when persisted two chars when displayed to indicate minimum reaction level
		/// </summary>
		private string _expectedMinimumReaction;
		/// <summary>
		/// Constant holding table's field names
		/// </summary>
		public const string REAGENTTYPECOLUMNS = "ReagentTypeId,ReagentTypeName";
		/// <summary>
		/// The result that they see with the plus sign
		/// </summary>
		private string _expectedMinimumReactionTestResultText;
		/// <summary>
		/// The test with name of the reagent
		/// </summary>
		private string _testWithName;
		/// <summary>
		/// The test with id of the reagent
		/// </summary>
		private int _testWithID;

		private System.Guid _reagentTestParameterGUID;
		private System.Guid _reagentTypeParameterGuid;

		/// <summary>
		/// Used to distiguish between the 4 screen cells
		/// </summary>
		private string _setVialID;
		/// <summary>
		/// dunno
		/// </summary>
		private string _reactivityPhaseCode;
		/// <summary>
		///     contains all valid reaction values, used to validate the set of this property
		/// </summary>
		
		private System.Collections.ArrayList _expectedMinReactions = null;

		#endregion


		#region Properties

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/4/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1006"> 
		///		<ExpectedInput>Valid int</ExpectedInput>
		///		<ExpectedOutput>Valid int</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1210"> 
		///		<ExpectedInput>int less than 1</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Reagent Type ID
		/// </summary>
		public int ReagentTypeId
		{
			get
			{
				return _reagentTypeID;
			}
			set
			{
				_reagentTypeID = value;
				//
				RuleBroken(MethodBase.GetCurrentMethod().Name,(_reagentTypeID<1));
			}
		}
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1520"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1221"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Name of test
		/// </summary>
		public string TestWithName
		{
			get
			{
				return this._testWithName;
			}
			set
			{
				this._testWithName = value;
			}
		}
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3444"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3445"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ReactivityPhaseCode
		/// </summary>
		public string ReactivityPhaseCode
		{
			get
			{
				return this._reactivityPhaseCode;
			}
			set
			{
				this._reactivityPhaseCode = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6534"> 
		///		<ExpectedInput>Valid integer</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6535"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// TestWithId
		/// </summary>
		public int TestWithId
		{
			get
			{
				return this._testWithID;
			}
			set
			{
				this._testWithID = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/4/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1007"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1214"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Reagent Type Name
		/// </summary>
		public string ReagentTypeName
		{
			get
			{
				return _reagentTypeName;
			}
			set
			{
				_reagentTypeName = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/4/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1008"> 
		///		<ExpectedInput>Valid short</ExpectedInput>
		///		<ExpectedOutput>Valid short</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1208"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Minimum Stock Level
		/// Implements BR_58.02
		/// </summary>
		public short MinimumStockLevel
		{
			//BR_58.02
			get
			{
				return _minimumStockLevel;
			}
			set
			{
				_minimumStockLevel = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/4/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1009"> 
		///		<ExpectedInput>Valid int</ExpectedInput>
		///		<ExpectedOutput>Valid int</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1215"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sort Column
		/// </summary>
		public short SortColumn
		{
			get
			{
				return _sortColumn;
			}
			set
			{
				_sortColumn = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1095"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1209"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// There are two kinds of indicators
		///on a reagent type.  This indicator says that 
		///the specific reagent type must be used, the other
		///indicator is a daily use indicator, which states that 
		///for a given day the reagent type must be used.
		/// </summary>
		public bool MustUseIndicator
		{
			get 
			{
				return this._mustUseIndicator;
			}
			set 
			{
				this._mustUseIndicator = value;
			}
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1519"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1167"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Indicator that implements
		/// BR 51.17  reagents exported with exp. min. reactions 
		/// of zero cannot be edited
		/// </summary>
		public bool CanChangeIndicator
		{
			get 
			{
				return this._canChangeIndicator;
			}
			set 
			{
				this._canChangeIndicator = value;
			}
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1097"> 
		///		<ExpectedInput>Valid bool</ExpectedInput>
		///		<ExpectedOutput>Valid bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1177"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///There are two kinds of indicators
		///on a reagent type.  This indicator says that 
		///the specific reagent type must be used, the other
		///indicator is a daily use indicator, which states that 
		///for a given day the reagent type must be used.
		/// </summary>
		public bool DailyUseIndicator
		{
			get 
			{
				return this._dailyUseIndicator;
			}
			set 
			{
				this._dailyUseIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6536"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6537"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SetVialId
		/// </summary>
		public string SetVialId
		{
			get 
			{
				return this._setVialID;
			}
			set 
			{
				this._setVialID = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1096"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1179"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///	This sets the minimum bounds
		///	that the expected reaction will be for a given reagent type	
		/// </summary>
		public string ExpectedMinimumReaction
		{
			get 
			{
				return this._expectedMinimumReaction;
			}
			set 
			{
				RuleBroken(MethodBase.GetCurrentMethod().Name, !_expectedMinReactions.Contains(value));
				this._expectedMinimumReaction = value;
			}
		}
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6538"> 
		///		<ExpectedInput>Valid GUID</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6539"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ReagentTypeParameterGuid
		/// </summary>
		public System.Guid ReagentTypeParameterGuid
		{
			get 
			{
				return this._reagentTypeParameterGuid;
			}
			set 
			{
				this._reagentTypeParameterGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6540"> 
		///		<ExpectedInput>Valid GUID</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6541"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ReagentTestParameterGuid
		/// </summary>
		public System.Guid ReagentTestParameterGuid
		{
			get 
			{
				return this._reagentTestParameterGUID;
			}
			set 
			{
				this._reagentTestParameterGUID = value;
			}
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/5/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1601"> 
		///		<ExpectedInput>A valid string</ExpectedInput>
		///		<ExpectedOutput>A valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1180"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This is the expected min reaction that they see with the plus sign
		/// </summary>
		public string ExpectedMinimumReactionTestResultText
		{
			get 
			{
				return this._expectedMinimumReactionTestResultText;
			}
			set 
			{
				this._expectedMinimumReactionTestResultText = value;
			}
		}

		#endregion


		#region Methods

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/4/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1010"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2020"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save Reagent Stock Levels
		/// </summary>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns></returns>
		public bool SaveReagentStockLevels(Common.UpdateFunction lastUpdateFunctionId)
		{
            //CR3508 this entire method was inside a try/catch block that absorbed the error and returned false - FrmMaintainReagentMinLevels.btnOK_Click() will catch now 
			bool bResult;

			DAL.ReagentType.SaveReagentStockLevels(this._reagentTypeParameterGuid, this._minimumStockLevel, Common.LogonUser.LogonUserName, base.RowVersion, lastUpdateFunctionId);
			bResult = true;

			return bResult;
		}

		///<Developers>
		///	<Developer>EDS EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/2/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5519"> 
		///		<ExpectedInput>UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5520"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InsertReagent
		/// </summary>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns></returns>
		public bool InsertReagent(Common.UpdateFunction lastUpdateFunctionId)
		{
            //CR3508 this entire method was inside a try/catch block that absorbed the error and returned false - FrmMaintainReagentMinLevels.btnOK_Click() will catch now 
			bool bResult;

			DAL.ReagentType.InsertReagent(this._reagentTypeID, this._minimumStockLevel, lastUpdateFunctionId);
			bResult = true;

			return bResult;
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/4/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1011"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1199"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// returns a DataTable of Reagent Types
		/// </summary>
		/// <returns></returns>
		public static DataTable GetReagentTypes()
		{
			//BR_58.01
			DataTable dtReagentTypes = DAL.ReagentType.GetReagentTypes();
			return dtReagentTypes;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/13/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7647"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7648"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get a list of reagent types and its associated antigen typing test ID.
		/// </summary>
		/// <returns></returns>
		public static DataTable GetAntigenTypingReagentTypes()
		{
			DataTable dtReagentTypes = DAL.ReagentType.GetAntigenTypingReagentTypes();
			return dtReagentTypes;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/23/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7649"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7650"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get a list of reagent types used for QC during antigen typing.
		/// Implements BR_41.25
		/// </summary>
		/// <returns></returns>
		public static DataTable GetAntigenTypingQCReagentTypes()
		{
			DataTable dtReagentTypes = DAL.ReagentType.GetAntigenTypingQCReagentTypes();
			return dtReagentTypes;
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/4/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1012"> 
		///		<ExpectedInput>Valid Division</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2023"> 
		///		<ExpectedInput>Invalid division</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a DataTable of existing minimum levels
		/// </summary>
		/// <param name="division"></param>
		/// <returns></returns>
		public static DataTable GetExistingMinimumLevels(string division)
		{
			DataTable dtReagentTypes = DAL.ReagentType.GetExistingMinimumLevels(division);
			return dtReagentTypes;
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1544"> 
		///		<ExpectedInput>None expected</ExpectedInput>
		///		<ExpectedOutput>A list of valid expected min. reactions that a reagent type can be changed to.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1198"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// returns an array list of expected minimum reactions that a reagent type can ve set to.
		/// </summary>
		/// <returns>ArrayList</returns>
		public static System.Collections.ArrayList GetCanChangeExpectedMinReactions()
		{
			System.Collections.ArrayList al = DAL.ReagentType.GetCanChangeExpectedMinReactions();
			return al;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1085"> 
		///		<ExpectedInput>Valid Division</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2024"> 
		///		<ExpectedInput>Invalid division</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// returns a DataTable of daily QC reagent types
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetDailyQCReagentTypes(string divisionCode)
		{
			DataTable dtReagentTypes = DAL.ReagentType.GetDailyQCReagentTypes(divisionCode);
			return dtReagentTypes;
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7651"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7652"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// returns a DataTable of daily QC reagent types
		/// </summary>
		/// <param name="dailyQcWorklistGuid"></param>
		/// <returns></returns>
		public static DataTable GetDailyQCReagentTypesForPartiallyCompletedQc(Guid dailyQcWorklistGuid)
		{
			DataTable dtReagentTypes = DAL.ReagentType.GetDailyQCReagentTypesForPartiallyCompletedQc(dailyQcWorklistGuid);
			return dtReagentTypes;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1385"> 
		///		<ExpectedInput>Commercial indicator flag set to true, Division Code  578</ExpectedInput>
		///		<ExpectedOutput>The data table containing the list of daily reagent types</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="1602"> 
		///		<ExpectedInput>Commercial indicator flag set to false, Division Code  578</ExpectedInput>
		///		<ExpectedOutput>The data table containing the list of daily reagent types</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2021"> 
		///		<ExpectedInput>Commercial indicator flag set to true, invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2022"> 
		///		<ExpectedInput>Commercial indicator flag set to false, invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of daily reagent types
		/// </summary>
		/// <param name="commercialIndicator"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static System.Collections.ArrayList GetReagentQCData(bool commercialIndicator, string divisionCode)
		{
			System.Collections.ArrayList alReagentTypeData;
			System.Data.DataTable dt = DAL.ReagentType.GetReagentQCData(divisionCode, commercialIndicator);
			alReagentTypeData = new System.Collections.ArrayList();
			foreach(System.Data.DataRow dtRow in dt.Rows)
			{
				alReagentTypeData.Add(new BOL.ReagentType(dtRow));
			}
			return alReagentTypeData;
		}



		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="1273"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1205"> 
		///		<ExpectedInput>Invalid data row</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.ReagentTypeId))
			{
				dtRow[TABLE.ReagentTestParameter.ReagentTypeId] = this.ReagentTypeId;
			}

			if(dtRow.Table.Columns.Contains(TABLE.ReagentType.ReagentTypeName))
			{
				dtRow[TABLE.ReagentType.ReagentTypeName] = this.ReagentTypeName;
			}


			//dtRow["MinimumStockLevel"] = this.MinimumStockLevel;
			if(dtRow.Table.Columns.Contains(TABLE.ReagentType.SortColumn))
			{
				if(this.SortColumn == short.MinValue)
				{
					dtRow[TABLE.ReagentType.SortColumn] = DBNull.Value;
				}
				else
				{
					dtRow[TABLE.ReagentType.SortColumn] = this.SortColumn;
				}

			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.DailyUseIndicator))
			{
				dtRow[TABLE.ReagentTestParameter.DailyUseIndicator] = this.DailyUseIndicator;
			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentType.MustUseIndicator))
			{
				dtRow[TABLE.ReagentType.MustUseIndicator] = this.MustUseIndicator;
			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.TestWithName))
			{
				dtRow[TABLE.ReagentTestParameter.TestWithName] = this.TestWithName;
			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.TestWithId))
			{
				dtRow[TABLE.ReagentTestParameter.TestWithId] = this.TestWithId;
			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.CanChangeIndicator))
			{
				dtRow[TABLE.ReagentTestParameter.CanChangeIndicator] = this.CanChangeIndicator;
			}
			if(dtRow.Table.Columns.Contains(TABLE.TestResult.TestResultText))
			{
				dtRow[TABLE.TestResult.TestResultText] = this.ExpectedMinimumReactionTestResultText;
			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.ExpectedMinimumReactions))
			{
				dtRow[TABLE.ReagentTestParameter.ExpectedMinimumReactions] = this.ExpectedMinimumReaction.Substring(0,1);
			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.ReagentTestParameterGuid))
			{
				dtRow[TABLE.ReagentTestParameter.ReagentTestParameterGuid] = this.ReagentTestParameterGuid;
			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.ReactivityPhaseCode))
			{
				dtRow[TABLE.ReagentTestParameter.ReactivityPhaseCode] =this.ReactivityPhaseCode;
			}
			if(dtRow.Table.Columns.Contains(TABLE.tmplReagentTestParameter.SetVialId))
			{
				dtRow[TABLE.tmplReagentTestParameter.SetVialId] = this.SetVialId;
			}
			if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.DivisionCode))
			{
				dtRow[TABLE.ReagentTestParameter.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			}

			return dtRow;
		}
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads an object from a data row
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if(this.VerifyMinColumnsInDataTable(REAGENTTYPECOLUMNS.Split(','), dtRow.Table))
			{
				this.ReagentTypeId = dtRow.IsNull(TABLE.ReagentType.ReagentTypeId) ? 0 : (int)dtRow[TABLE.ReagentType.ReagentTypeId];
				this.ReagentTypeName = dtRow.IsNull(TABLE.ReagentType.ReagentTypeName) ? string.Empty : dtRow[TABLE.ReagentType.ReagentTypeName].ToString();
				if(dtRow.Table.Columns.Contains(TABLE.ReagentTypeParameter.MinimumStockLevel))
				{
					this.MinimumStockLevel= dtRow.IsNull(TABLE.ReagentTypeParameter.MinimumStockLevel) ? short.MinValue : (short) dtRow[TABLE.ReagentTypeParameter.MinimumStockLevel];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentType.SortColumn))
				{
					this.SortColumn = dtRow.IsNull(TABLE.ReagentType.SortColumn) ? short.MinValue : (short)dtRow[TABLE.ReagentType.SortColumn];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.DailyUseIndicator))
				{
					this.DailyUseIndicator =dtRow.IsNull(TABLE.ReagentTestParameter.DailyUseIndicator) ? false : (bool)dtRow[TABLE.ReagentTestParameter.DailyUseIndicator];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentType.MustUseIndicator))
				{
					this.MustUseIndicator = dtRow.IsNull(TABLE.ReagentType.MustUseIndicator) ? false : (bool)dtRow[TABLE.ReagentType.MustUseIndicator];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.TestWithName))
				{
					this.TestWithName = dtRow.IsNull(TABLE.ReagentTestParameter.TestWithName) ? string.Empty : (string)dtRow[TABLE.ReagentTestParameter.TestWithName];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.TestWithId))
				{
					this.TestWithId = dtRow.IsNull(TABLE.ReagentTestParameter.TestWithId) ? int.MinValue : (int)dtRow[TABLE.ReagentTestParameter.TestWithId];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.CanChangeIndicator))
				{
					this.CanChangeIndicator = dtRow.IsNull(TABLE.ReagentTestParameter.CanChangeIndicator) ? false : (bool)dtRow[TABLE.ReagentTestParameter.CanChangeIndicator];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.ExpectedMinimumReactions))
				{
					this.ExpectedMinimumReaction = dtRow.IsNull(TABLE.ReagentTestParameter.ExpectedMinimumReactions) ? string.Empty : (string)dtRow[TABLE.ReagentTestParameter.ExpectedMinimumReactions];
				}
				if(dtRow.Table.Columns.Contains(TABLE.TestResult.TestResultText))
				{
					this.ExpectedMinimumReactionTestResultText = dtRow.IsNull(TABLE.TestResult.TestResultText) ? string.Empty : (string)dtRow[TABLE.TestResult.TestResultText];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.ReactivityPhaseCode))
				{
					this.ReactivityPhaseCode = dtRow.IsNull(TABLE.ReagentTestParameter.ReactivityPhaseCode) ? string.Empty : (string)dtRow[TABLE.ReagentTestParameter.ReactivityPhaseCode];
				}
				if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.ReagentTestParameterGuid))
				{
					this.ReagentTestParameterGuid = dtRow.IsNull(TABLE.ReagentTestParameter.ReagentTestParameterGuid) ? System.Guid.NewGuid() : (System.Guid)dtRow[TABLE.ReagentTestParameter.ReagentTestParameterGuid];
				}
				if(dtRow.Table.Columns.Contains(TABLE.tmplReagentTestParameter.SetVialId))
				{
					this.SetVialId = dtRow.IsNull(TABLE.tmplReagentTestParameter.SetVialId) ? System.String.Empty : (System.String)dtRow[TABLE.tmplReagentTestParameter.SetVialId];
				}
//				if(dtRow.Table.Columns.Contains(TABLE.ReagentTestParameter.DivisionCode))
//				{
//					this.DivisionCode = dtRow.IsNull(TABLE.ReagentTestParameter.DivisionCode) ? System.String.Empty : (System.String)dtRow[TABLE.ReagentTestParameter.DivisionCode];
//				}
			}
			else
			{
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					"load data row from reagent type").ResString);	
			}
		}

		
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="1272"> 
		///		<ExpectedInput>string array of column names, Valid data table</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1243"> 
		///		<ExpectedInput>invalid data table</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks the columns in the data table against a string array of column names
		/// </summary>
		public bool VerifyMinColumnsInDataTable(string [] colNames, System.Data.DataTable dtTable)
		{
			return Common.Utility.VerifyMinColumnsInDataTable(colNames, dtTable);
		}
		
		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7653"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7654"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// //BR 51.08:
		/// If Screening Cell 3 (SC3) and/or (SC4) are set to yes "required" as part of the
		/// configuration of the antibody screening cell set in UC_51, then the template 
		/// the 3rd and 4th Screening Cell (SC) are enabled on the antibody screen data grid. 
		/// Present test options template.  
		/// </summary>
		/// <returns></returns>
		public bool CheckScreeningCell3And4()
		{
			const int SC = 63; 
			const string SCVIALID3 = "3";
			const string SCVIALID4 = "4";


			if((System.Convert.ToInt32(this._reagentTypeID.ToString()) == SC) &&
				(this.SetVialId == SCVIALID3 || this.SetVialId == SCVIALID4))																		
			{
				return true;
			}
			else
			{
				return false;
			}
		}

		#endregion
	}
}




